/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { defineStore } from "pinia";
import type { StrategyItem, CheckItems } from "@/types/continous-profiling";
import type { EBPFTaskList, EBPFProfilingSchedule, AnalyzationTrees } from "@/types/ebpf";
import type { Instance } from "@/types/selector";
import { store } from "@/store";
import graphql from "@/graphql";
import type { MonitorInstance, MonitorProcess } from "@/types/continous-profiling";
import { dateFormat } from "@/utils/dateFormat";

interface ContinousProfilingState {
  strategyList: Array<StrategyItem>;
  selectedStrategy: Nullable<StrategyItem>;
  taskList: Array<EBPFTaskList>;
  selectedTask: Nullable<EBPFTaskList>;
  errorTip: string;
  errorReason: string;
  instances: Instance[];
  instance: Nullable<Instance>;
  eBPFSchedules: EBPFProfilingSchedule[];
  currentSchedule: Nullable<EBPFProfilingSchedule>;
  analyzeTrees: AnalyzationTrees[];
  ebpfTips: string;
  aggregateType: string;
  instancesLoading: boolean;
  policyLoading: boolean;
}

export const continousProfilingStore = defineStore({
  id: "continousProfiling",
  state: (): ContinousProfilingState => ({
    strategyList: [],
    selectedStrategy: null,
    taskList: [],
    selectedTask: null,
    errorReason: "",
    errorTip: "",
    ebpfTips: "",
    instances: [],
    eBPFSchedules: [],
    currentSchedule: null,
    analyzeTrees: [],
    aggregateType: "COUNT",
    instance: null,
    instancesLoading: false,
    policyLoading: false,
  }),
  actions: {
    setSelectedStrategy(task: Nullable<StrategyItem>) {
      this.selectedStrategy = task || {};
    },
    setselectedTask(task: Recordable<EBPFTaskList>) {
      this.selectedTask = task || {};
    },
    setCurrentSchedule(s: EBPFProfilingSchedule) {
      this.currentSchedule = s;
    },
    setAnalyzeTrees(tree: AnalyzationTrees[]) {
      this.analyzeTrees = tree;
    },
    setCurrentInstance(instance: Nullable<Instance>) {
      this.instance = instance;
    },
    async setContinuousProfilingPolicy(
      serviceId: string,
      targets: {
        targetType: string;
        checkItems: CheckItems[];
      }[],
    ) {
      const response = await graphql.query("editStrategy").params({
        request: {
          serviceId,
          targets,
        },
      });

      if (response.errors) {
        return response;
      }

      return response;
    },
    async getStrategyList(params: { serviceId: string }) {
      if (!params.serviceId) {
        return new Promise((resolve) => resolve({}));
      }
      this.policyLoading = true;
      const response = await graphql.query("getStrategyList").params(params);

      this.policyLoading = false;
      if (response.errors) {
        return response;
      }
      const list = response.data.strategyList || [];
      if (!list.length) {
        this.taskList = [];
        this.instances = [];
        this.instance = null;
      }
      const arr = list.length ? response.data.strategyList : [{ type: "", checkItems: [{ type: "" }] }];
      this.strategyList = arr.map((d: StrategyItem, index: number) => {
        return {
          ...d,
          id: index,
        };
      });
      this.setSelectedStrategy(this.strategyList[0]);
      if (!this.selectedStrategy.type) {
        return response;
      }
      this.getMonitoringInstances(params.serviceId);
      return response;
    },
    async getMonitoringInstances(serviceId: string) {
      this.instancesLoading = true;
      if (!serviceId) {
        return null;
      }
      const response = await graphql.query("getMonitoringInstances").params({
        serviceId,
        target: this.selectedStrategy.type,
      });
      this.instancesLoading = false;
      if (response.errors) {
        return response;
      }
      this.instances = (response.data.instances || [])
        .map((d: MonitorInstance) => {
          const processes = (d.processes || [])
            .sort((c: MonitorProcess, d: MonitorProcess) => d.lastTriggerTimestamp - c.lastTriggerTimestamp)
            .map((p: MonitorProcess) => {
              return {
                ...p,
                lastTriggerTime: d.lastTriggerTimestamp ? dateFormat(d.lastTriggerTimestamp) : "",
                labels: p.labels.join("; "),
              };
            });

          return {
            ...d,
            processes,
            lastTriggerTime: d.lastTriggerTimestamp ? dateFormat(d.lastTriggerTimestamp) : "",
          };
        })
        .sort((a: MonitorInstance, b: MonitorInstance) => b.lastTriggerTimestamp - a.lastTriggerTimestamp);
      this.instance = this.instances[0] || null;
      return response;
    },
  },
});

export function useContinousProfilingStore() {
  return continousProfilingStore(store);
}
